<?php

namespace App\Exports;

use App\Models\Financement;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class FinancementsExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles
{
    protected $filters;

    public function __construct(array $filters = [])
    {
        $this->filters = $filters;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        $query = Financement::query();

        // Appliquer les filtres
        if (!empty($this->filters['statut'])) {
            $query->where('statut', $this->filters['statut']);
        }

        if (!empty($this->filters['date_debut'])) {
            $query->whereDate('created_at', '>=', $this->filters['date_debut']);
        }

        if (!empty($this->filters['date_fin'])) {
            $query->whereDate('created_at', '<=', $this->filters['date_fin']);
        }

        return $query->get();
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            'ID',
            'Référence',
            'Type de structure',
            'Nom de la structure',
            'Personne habilitée',
            'Email',
            'Téléphone',
            'Titre du projet',
            'Résumé',
            'Période de mise en œuvre',
            'Organisations partenaires',
            'Coût total',
            'Montant sollicité',
            'Statut',
            'Date de création',
            'Date de dernière mise à jour',
        ];
    }

    /**
     * @param mixed $financement
     *
     * @return array
     */
    public function map($financement): array
    {
        return [
            $financement->id,
            $financement->reference,
            $financement->type_structure,
            $financement->nom_structure,
            $financement->personne_habilitee,
            $financement->email,
            $financement->telephone,
            $financement->titre_projet,
            $financement->resume,
            $financement->periode_mise_oeuvre,
            $financement->organisations_partenaires,
            number_format($financement->cout_total, 0, ',', ' ') . ' FCFA',
            number_format($financement->montant_sollicite, 0, ',', ' ') . ' FCFA',
            ucfirst($financement->statut),
            $financement->created_at->format('d/m/Y H:i'),
            $financement->updated_at->format('d/m/Y H:i'),
        ];
    }

    /**
     * @param Worksheet $sheet
     */
    public function styles(Worksheet $sheet)
    {
        return [
            // Style de l'en-tête
            1 => [
                'font' => ['bold' => true],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'D9EAD3']
                ]
            ],
        ];
    }
}
