<?php

namespace App\Http\Controllers;

use App\Models\User;
use Spatie\Activitylog\Models\Activity;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ActivityLogController extends Controller
{
    /**
     * Display a listing of the activity logs.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $query = Activity::with('causer')
            ->latest();

        // Search functionality
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('description', 'like', "%{$search}%")
                  ->orWhere('event', 'like', "%{$search}%")
                  ->orWhereHas('causer', function($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%")
                        ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        // Filter by user
        if ($request->has('user_id') && !empty($request->user_id)) {
            $query->where('causer_id', $request->user_id);
        }

        // Filter by model type
        if ($request->has('subject_type') && !empty($request->subject_type)) {
            $query->where('subject_type', urldecode($request->subject_type));
        }

        // Filter by date range
        if ($request->has('date_range') && !empty($request->date_range)) {
            $dates = explode(' au ', $request->date_range);
            $startDate = $dates[0] . ' 00:00:00';
            $endDate = isset($dates[1]) ? $dates[1] . ' 23:59:59' : $dates[0] . ' 23:59:59';
            
            $query->whereBetween('created_at', [$startDate, $endDate]);
        }

        $activities = $query->paginate(20);
        $users = User::orderBy('nom')->orderBy('prenom')->get();
        
        // Get all model types that exist in the activity log
        $modelTypes = Activity::select('subject_type')
            ->whereNotNull('subject_type')
            ->distinct()
            ->pluck('subject_type')
            ->mapWithKeys(function($type) {
                return [$type => class_basename($type)];
            })
            ->unique();

        return view('activity-logs.index', compact('activities', 'users', 'modelTypes'));
    }

    /**
     * Display the specified activity log.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $activity = Activity::with(['causer', 'subject'])->findOrFail($id);
        
        return view('activity-logs.show', compact('activity'));
    }
}
