<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use App\Http\Requests\StoreRoleRequest;
use App\Http\Requests\UpdateRoleRequest;

class RoleController extends Controller
{
    public static function middleware(): array
    {
        return [
            'auth',
            new \Illuminate\Routing\Controllers\Middleware('permission:role-list|role-create|role-edit|role-delete', ['only' => ['index','show']]),
            new \Illuminate\Routing\Controllers\Middleware('permission:role-create', ['only' => ['create','store']]),
            new \Illuminate\Routing\Controllers\Middleware('permission:role-edit', ['only' => ['edit','update']]),
            new \Illuminate\Routing\Controllers\Middleware('permission:role-delete', ['only' => ['destroy']]),
        ];
    }
    
    public function index(Request $request)
    {
        $roles = Role::with('permissions')
            ->when($request->search, function($query, $search) {
                $query->where('name', 'like', "%{$search}%");
            })
            ->paginate(10);
            
        return view('admin.roles.index', compact('roles'));
    }

    public function create()
    {
        $permissions = Permission::all();
        return view('admin.roles.create', compact('permissions'));
    }

    public function store(StoreRoleRequest $request)
    {
        $role = Role::create(['name' => $request->name]);
        
        if($request->permissions) {
            $role->syncPermissions($request->permissions);
        }

        return redirect()->route('roles.index')
                        ->with('success', 'Rôle créé avec succès.');
    }

    public function show(Role $role)
    {
        $role->load('permissions', 'users');
        return view('admin.roles.show', compact('role'));
    }

    public function edit(Role $role)
    {
        $permissions = Permission::all();
        $rolePermissions = $role->permissions->pluck('name')->toArray();
        
        return view('admin.roles.edit', compact('role', 'permissions', 'rolePermissions'));
    }

    public function update(UpdateRoleRequest $request, Role $role)
    {
        $role->update(['name' => $request->name]);
        $role->syncPermissions($request->permissions ?? []);

        return redirect()->route('roles.index')
                        ->with('success', 'Rôle mis à jour avec succès.');
    }

    public function destroy(Role $role)
    {
        if($role->users()->count() > 0) {
            return redirect()->route('roles.index')
                           ->with('error', 'Impossible de supprimer ce rôle car il est assigné à des utilisateurs.');
        }

        $role->delete();

        return redirect()->route('roles.index')
                        ->with('success', 'Rôle supprimé avec succès.');
    }
}
