<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use App\Http\Requests\StoreUserRequest;
use App\Http\Requests\UpdateUserRequest;
use Illuminate\Foundation\Mix;
use Illuminate\Routing\Controllers\Middleware;

class UserController extends Controller
{

    public static function middleware(): array
    {
        return [
            'auth',
            new Middleware('permission:user-list|user-create|user-edit|user-delete', ['only' => ['index','show']]),
            new Middleware('permission:user-create', ['only' => ['create','store']]),
            new Middleware('permission:user-edit', ['only' => ['edit','update']]),
            new Middleware('permission:user-delete', ['only' => ['destroy']]),
        ];
    }
    public function __construct()
    {

        $this->middleware('permission:user-list|user-create|user-edit|user-delete', ['only' => ['index','show']]);
        $this->middleware('permission:user-create', ['only' => ['create','store']]);
        $this->middleware('permission:user-edit', ['only' => ['edit','update']]);
        $this->middleware('permission:user-delete', ['only' => ['destroy']]);
    }

    public function index(Request $request)
    {
        $users = User::with('roles')
            ->when($request->search, function($query, $search) {
                $query->where('nom', 'like', "%{$search}%")
                    //   ->orWhere('prenom', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%");
            })
            ->when($request->role, function($query, $role) {
                $query->whereHas('roles', function($q) use ($role) {
                    $q->where('name', $role);
                });
            })
            ->when($request->status !== null, function($query) use ($request) {
                $query->where('is_active', $request->status);
            })
            ->paginate(10);

        $roles = Role::all();
        
        return view('admin.users.index', compact('users', 'roles'));
    }

    public function create()
    {
        $roles = Role::all();
        return view('admin.users.create', compact('roles'));
    }

    public function store(StoreUserRequest $request)
    {
        $user = User::create([
            'nom' => $request->nom,
            'prenom' => $request->prenom,
            'name' => $request->nom . ' ' . $request->prenom,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'is_active' => $request->has('is_active')
        ]);

        if($request->roles) {
            $user->assignRole($request->roles);
        }

        return redirect()->route('users.index')
                        ->with('success', 'Utilisateur créé avec succès.');
    }

    public function show(User $user)
    {
        return view('admin.users.show', compact('user'));
    }

    public function edit(User $user)
    {
        $roles = Role::all();
        $userRoles = $user->roles->pluck('name')->toArray();
        
        return view('admin.users.edit', compact('user', 'roles', 'userRoles'));
    }

    public function update(UpdateUserRequest $request, User $user)
    {
        $data = [
            'nom' => $request->nom,
            'prenom' => $request->prenom,
            'name' => $request->nom . ' ' . $request->prenom,
            'email' => $request->email,
            'is_active' => $request->has('is_active'),
        ];
        // dd($data);
        if($request->filled('password')) {
            $data['password'] = Hash::make($request->password);
        }
        try {
            $user->update($data);
            $user->syncRoles($request->roles ?? []);
        } catch (\Exception $e) {
            dd($e->getMessage());
            // return redirect()->back()->withErrors(['roles' => 'Erreur lors de la mise à jour des rôles.'])->withInput();
        }
    

        return redirect()->route('users.index')
                        ->with('success', 'Utilisateur mis à jour avec succès.');
    }

    public function destroy(User $user)
    {
        if($user->id == auth()->id()) {
            return redirect()->route('users.index')
                           ->with('error', 'Vous ne pouvez pas supprimer votre propre compte.');
        }

        $user->delete();

        return redirect()->route('users.index')
                        ->with('success', 'Utilisateur supprimé avec succès.');
    }
}
