<?php

namespace App\Http\Controllers;

use App\Models\Plainte;
use App\Models\Financement;
use App\Models\Resolution;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        // Get counts for dashboard cards
        $stats = [
            'total_plaintes' => Plainte::count(),
            'plaintes_en_attente' => Plainte::where('statut', 'en_attente')->count(),
            'plaintes_en_cours' => Plainte::where('statut', 'en_cours')->count(),
            'plaintes_traitees' => Plainte::whereIn('statut', ['traitee', 'cloturee'])->count(),
            'total_financements' => Financement::count(),
            'financements_en_attente' => Financement::where('statut', 'en_attente')->count(),
            'financements_approuves' => Financement::where('statut', 'approuve')->count(),
            'financements_rejetes' => Financement::where('statut', 'rejete')->count(),
        ];

        // Get recent activities
        $recentActivities = $this->getRecentActivities();

        // Get statistics by region
        $plaintesByRegion = $plainte = Plainte::join('plaignants', 'plaintes.plaignant_id', '=', 'plaignants.id')
            ->groupBy('nom')
            ->select('nom', DB::raw('COUNT(plaintes.id) as total'))
            ->orderBy('total', 'desc')
            ->get();

        //Plaintes par sensibilité
        $plaintesBySensibilite = [
            'total' => Plainte::count(),
            'sensible' => Plainte::where('niveau_sensibilite', 'sensible')->count(),
            'non_sensible' => Plainte::where('niveau_sensibilite', 'non_sensible')->count(),
        ];
        $plaintesByMois = Plainte::select(DB::raw('EXTRACT(MONTH FROM created_at) as mois'), DB::raw('COUNT(*) as total'))
            ->groupBy('mois')
            ->orderBy('mois', 'asc')
            ->get();
        $financementsByMois = Financement::select(DB::raw('EXTRACT(MONTH FROM created_at) as mois'), DB::raw('COUNT(*) as total'))
            ->groupBy('mois')
            ->orderBy('mois', 'asc')
            ->get();
        return view('dashboard.index', compact('stats', 'recentActivities', 'plaintesByRegion', 'plaintesBySensibilite', 'plaintesByMois', 'financementsByMois'));
    }

    private function getRecentActivities()
    {
        $plaintes = Plainte::latest()
            ->take(5)
            ->get()
            ->map(function($item) {
                return [
                    'type' => 'plainte',
                    'id' => $item->id,
                    'titre' => $item->objet,
                    'date' => $item->created_at,
                    'statut' => $item->statut,
                    'user' => $item->user->fullName ?? 'Anonyme',
                    'icon' => 'bi bi-exclamation-circle',
                    'color' => 'primary'
                ];
            });

        $financements = Financement::latest()
            ->take(5)
            ->get()
            ->map(function($item) {
                return [
                    'type' => 'financement',
                    'id' => $item->id,
                    'titre' => $item->intitule_activite,
                    'date' => $item->created_at,
                    'statut' => $item->statut,
                    'user' => $item->user->name ?? 'Anonyme',
                    'icon' => 'fas fa-hand-holding-usd',
                    'color' => 'success'
                ];
            });

        // Merge and sort by date
        $activities = $plaintes->merge($financements)
            ->sortByDesc('date')
            ->take(5);

        return $activities;
    }
}