<?php

namespace App\Http\Controllers;

use App\Http\Requests\CreateFinancementRequest;
use App\Mail\FinancementAccuseReceptionMail;
use App\Mail\FinancementStatusChangeMail;
use App\Models\Financement;
use App\Models\Media;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;

class FinancementController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Financement::query();

        // Filtres
        if ($request->statut) {
            $query->where('statut', $request->statut);
        }
        
        if ($request->region) {
            $query->where('region', $request->region);
        }
        
        if ($request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('reference', 'like', "%{$search}%")
                  ->orWhere('nom_structure', 'like', "%{$search}%")
                  ->orWhere('adresse', 'like', "%{$search}%");
            });
        }
        
        $financements = $query->latest()->paginate(15);
        $stats = [
            'total' => Financement::count(),
            'en_attente' => Financement::where('statut', 'en_attente')->count(),
            'approuve' => Financement::where('statut', 'approuve')->count(),
            'rejete' => Financement::where('statut', 'rejete')->count(),
        ];
        $regions = [
            'Boucle du Mouhoun', 'Cascades', 'Centre', 'Centre-Est', 'Centre-Nord',
            'Centre-Ouest', 'Centre-Sud', 'Est', 'Hauts-Bassins', 'Nord',
            'Plateau-Central', 'Sahel', 'Sud-Ouest'
        ];
        return view('financements.index', compact('financements','regions','stats'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $regions = [
            'Boucle du Mouhoun', 'Cascades', 'Centre', 'Centre-Est', 'Centre-Nord',
            'Centre-Ouest', 'Centre-Sud', 'Est', 'Hauts-Bassins', 'Nord',
            'Plateau-Central', 'Sahel', 'Sud-Ouest'
        ];
        
        return view('financements.formulaire', compact('regions'));
    }

    public function formulaire()
    {
        $regions = [
            'Boucle du Mouhoun', 'Cascades', 'Centre', 'Centre-Est', 'Centre-Nord',
            'Centre-Ouest', 'Centre-Sud', 'Est', 'Hauts-Bassins', 'Nord',
            'Plateau-Central', 'Sahel', 'Sud-Ouest'
        ];
        
        return view('financements.formulaire', compact('regions'));
    }
    /**
     * Store a newly created resource in storage.
     */
    public function store(CreateFinancementRequest $request)
    {

        try {
           
            $validated = $request->validated();
            // Create new financement
            $validated['statut'] = 'en_attente';
            $financement = Financement::create($validated);

            if ($request->hasFile('piece_jointes')) {
                foreach ($request->file('piece_jointes') as $file) {
                    $pj = $this->uploadFile($file, 'documents/piece_jointes');
                    $financement->media()->create([
                        'nom' => $file->getClientOriginalName(),
                        'type' => $file->extension(),
                        'chemin'=>$pj,
                        'taille'=>$file->getSize(),

                    ]);
                }
            }
            
            // Envoyer l'accusé de réception par email
            if ($request->has('email')) {
                try {
                    Mail::to($request->email)
                        ->send(new FinancementAccuseReceptionMail($financement));
                } catch (\Exception $e) {
                    dd($e);
                    Log::error('Erreur lors de l\'envoi de l\'accusé de réception: ' . $e->getMessage());
                }
            }

            return redirect()
                ->route('financements.suivi-financement', $financement->reference)
                ->with('success', 'Votre demande de financement a été soumise avec succès. Numéro de référence: ' . $financement->reference);

        } catch (\Exception $e) {
            dd($e);
            Log::error('Erreur lors de la création de la demande de financement: ' . $e->getMessage());
            
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Une erreur est survenue lors de la soumission de votre demande. Veuillez réessayer.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Financement $financement)
    {
        return view('financements.show', compact('financement'));
    }

    public function suiviFinancement($reference)
    {
        $financement = Financement::where('reference', $reference)->first();
        if (!$financement) {
            abort(404);
        }
        return view('financements.suivi-financement', compact('financement'));
    }
    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Financement $financement)
    {
        $regions = [
            'Boucle du Mouhoun', 'Cascades', 'Centre', 'Centre-Est', 'Centre-Nord',
            'Centre-Ouest', 'Centre-Sud', 'Est', 'Hauts-Bassins', 'Nord',
            'Plateau-Central', 'Sahel', 'Sud-Ouest'
        ];
        
        return view('financements.edit', compact('financement', 'regions'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Financement $financement)
    {
        $validator = Validator::make($request->all(), [
            'nom_structure' => 'required|string|max:255',
            'statut_juridique' => 'required|string|in:association,ong,cooperative,entreprise,groupement,autre',
            'adresse' => 'required|string|max:1000',
            'numero_ifu' => 'required|string|max:50',
            'numero_rccm' => 'required|string|max:50',
            'region' => 'required|string|max:100',
            'commune' => 'required|string|max:100',
            'personne_habilitee' => 'required|string|max:255',
            'telephone' => 'required|string|max:20',
            'email' => 'required|email|max:100',
            'intitule_activite' => 'required|string|max:255',
            'resume_activite' => 'required|string|max:5000',
            'periode_mise_en_oeuvre' => 'required|string|max:100',
            'organisations_partenaires' => 'nullable|string|max:500',
            'cout_total' => 'required|numeric|min:0',
            'montant_demande' => 'required|numeric|min:0|lte:cout_total',
            'duree_remboursement' => 'nullable|string|max:100',
            'sources_financement_complementaires' => 'nullable|string|max:1000',
            'document_projet' => 'nullable|file|mimes:pdf,doc,docx|max:10240',
            'budget_detail' => 'nullable|file|mimes:pdf,xls,xlsx|max:10240',
            'chronogramme' => 'nullable|file|mimes:pdf,xls,xlsx|max:10240',
            'cv_responsables' => 'nullable|file|mimes:pdf,doc,docx|max:10240',
            'autres_documents' => 'nullable|file|mimes:pdf,doc,docx,xls,xlsx,jpg,jpeg,png|max:10240',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput()
                ->with('error', 'Veuillez corriger les erreurs dans le formulaire.');
        }

        try {
            $data = $request->except([
                'document_projet', 'budget_detail', 'chronogramme', 
                'cv_responsables', 'autres_documents'
            ]);

            

            $financement->update($data);

            return redirect()
                ->route('financements.show', $financement->id)
                ->with('success', 'La demande de financement a été mise à jour avec succès.');

        } catch (\Exception $e) {
            Log::error('Erreur lors de la mise à jour de la demande de financement: ' . $e->getMessage());
            
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Une erreur est survenue lors de la mise à jour de votre demande. Veuillez réessayer.');
        }
    }

    /**
     * Update the status of the specified resource.
     */
    public function updateStatus(Request $request, Financement $financement)
    {
        $request->validate([
            'statut' => 'required|in:en_attente,archive,en_cours,approuve,rejete',
            'commentaire' => 'nullable|string|max:1000',
        ]);

        $financement->update([
            'statut' => $request->statut,
            'commentaire' => $request->commentaire,
        ]);

        // Envoyer l'accusé de réception par email
        if ($financement->email) {
            try {
                Mail::to($financement->email)
                    ->send(new FinancementStatusChangeMail($financement));
            } catch (\Exception $e) {
                dd($e);
                Log::error('Erreur lors de l\'envoi du statut de la demande de financement: ' . $e->getMessage());
            }
        }
        
        return redirect()
            ->back()
            ->with('success', 'Le statut de la demande a été mis à jour avec succès.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Financement $financement)
    {
        try {
            // Delete associated files
            

            $financement->delete();

            return redirect()
                ->route('financements.index')
                ->with('success', 'La demande de financement a été supprimée avec succès.');

        } catch (\Exception $e) {
            Log::error('Erreur lors de la suppression de la demande de financement: ' . $e->getMessage());
            
            return redirect()
                ->back()
                ->with('error', 'Une erreur est survenue lors de la suppression de la demande. Veuillez réessayer.');
        }
    }

    /**
     * Download a file associated with a financement.
     */
    public function downloadFile(Media $media)
    {

        // $filePath = "documents/piece_jointes/";
        $fileName = $media->chemin;

        return Storage::download('storage/'.$fileName);
    }

    /**
     * Upload a file to storage.
     */
    private function uploadFile($file, $directory)
    {
        if (!$file) {
            return null;
        }

        $fileName = Str::random(40) . '.' . $file->getClientOriginalExtension();
        return $file->storeAs($directory, $fileName, 'public');
    }

    /**
     * Delete a file from storage.
     */
    private function deleteFile($filePath)
    {
        if ($filePath && Storage::exists($filePath)) {
            Storage::delete($filePath);
        }
    }
}
