<?php

namespace App\Http\Controllers;

use App\Models\Plainte;
use Illuminate\Http\Request;

class ReportController extends Controller
{
    public function dashboard()
    {
        $stats = [
            'total_plaintes' => Plainte::count(),
            'plaintes_mois' => Plainte::whereMonth('date_reception', now()->month)->count(),
            'taux_resolution' => $this->calculateResolutionRate(),
            'delai_moyen' => $this->calculateAverageResolutionTime(),
            'par_type' => Plainte::selectRaw('type_plainte, count(*) as total')
                                   ->groupBy('type_plainte')
                                   ->pluck('total', 'type_plainte'),
            'par_statut' => Plainte::selectRaw('statut, count(*) as total')
                                     ->groupBy('statut')
                                     ->pluck('total', 'statut'),
            'par_sensibilite' => Plainte::selectRaw('niveau_sensibilite, count(*) as total')
                                         ->groupBy('niveau_sensibilite')
                                         ->pluck('total', 'niveau_sensibilite'),
        ];

        return view('reports.dashboard', compact('stats'));
    }

    public function monthlyReport(Request $request)
    {
        $month = $request->get('month', now()->format('Y-m'));
        [$year, $monthNum] = explode('-', $month);

        $plaintes = Plainte::whereYear('date_reception', $year)
                              ->whereMonth('date_reception', $monthNum)
                              ->with(['plaignant', 'assignedTo'])
                              ->get();

        $stats = [
            'total' => $plaintes->count(),
            'resolues' => $plaintes->where('statut', 'resolu')->count(),
            'en_cours' => $plaintes->whereIn('statut', ['recu', 'en_examen', 'en_enquete'])->count(),
            'sensibles' => $plaintes->where('niveau_sensibilite', 'sensible')->count(),
            'par_canal' => $plaintes->groupBy('canal_reception')->map->count(),
            'delai_moyen' => $plaintes->avg('delai_traitement'),
        ];

        return view('reports.monthly', compact('plaintes', 'stats', 'month'));
    }

    private function calculateResolutionRate()
    {
        $total = Plainte::count();
        if ($total === 0) return 0;
        
        $resolved = Plainte::whereIn('statut', ['resolu', 'clos'])->count();
        return round(($resolved / $total) * 100, 1);
    }

    private function calculateAverageResolutionTime()
    {

        return Plainte::whereNotNull('date_resolution')
    ->selectRaw("AVG((date_resolution - date_reception)) as avg_days")
    ->value('avg_days') ?? 0;
    }
}
