<?php

namespace App\Http\Controllers;

use App\Models\Plainte;
use App\Models\Resolution;
use App\Mail\ResolutionProposedMail;
use App\Mail\ResolutionStatusUpdatedMail;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class ResolutionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Plainte $plainte)
    {
        $resolutions = $plainte->resolutions()
            ->with(['proposer', 'media'])
            ->latest()
            ->paginate(10);

        return view('resolutions.index', compact('plainte', 'resolutions'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Plainte $plainte)
    {
        return view('resolutions.create', compact('plainte'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request, Plainte $plainte)
    {
        $request->validate([
            'resolution' => 'required|string|min:10',
            'fichiers.*' => 'nullable|file|max:10240|mimes:jpg,jpeg,png,pdf,doc,docx,xls,xlsx',
        ]);

        DB::beginTransaction();
        try {
            $resolution = $plainte->resolutions()->create([
                'resolution' => $request->resolution,
                'statut' => 'en_attente',
                'date_proposition' => now(),
                'propose_par' => auth()->id(),
            ]);

            // Handle file uploads
            if ($request->hasFile('fichiers')) {
                foreach ($request->file('fichiers') as $file) {
                    $path = $file->store('resolutions/' . $plainte->id, 'public');
                    
                    $resolution->media()->create([
                        'file_name' => $file->getClientOriginalName(),
                        'file_path' => $path,
                        'mime_type' => $file->getMimeType(),
                        'size' => $file->getSize(),
                    ]);
                }
            }

            // Send email to the complainant
            if ($plainte->plaignant && $plainte->plaignant->email) {
                Mail::to($plainte->plaignant->email)
                    ->send(new ResolutionProposedMail($resolution));
            }

            DB::commit();

            return redirect()
                ->route('plaintes.show', $plainte)
                ->with('success', 'La proposition de résolution a été envoyée avec succès.');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Erreur lors de la création de la résolution: ' . $e->getMessage());
            
            return redirect()
                ->back()
                ->with('error', 'Une erreur est survenue lors de l\'envoi de la proposition de résolution.')
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Plainte $plainte, Resolution $resolution)
    {
        $this->authorize('view', $resolution);
        
        $resolution->load(['proposer', 'media']);
        
        return view('resolutions.show', compact('plainte', 'resolution'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Plainte $plainte, Resolution $resolution)
    {
        $this->authorize('update', $resolution);
        
        return view('resolutions.edit', compact('plainte', 'resolution'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Plainte $plainte, Resolution $resolution)
    {
        $this->authorize('update', $resolution);
        
        $request->validate([
            'resolution' => 'required|string|min:10',
            'fichiers.*' => 'nullable|file|max:10240|mimes:jpg,jpeg,png,pdf,doc,docx,xls,xlsx',
        ]);

        DB::beginTransaction();
        try {
            $resolution->update([
                'resolution' => $request->resolution,
            ]);

            // Handle file uploads
            if ($request->hasFile('fichiers')) {
                foreach ($request->file('fichiers') as $file) {
                    $path = $file->store('resolutions/' . $plainte->id, 'public');
                    
                    $resolution->media()->create([
                        'file_name' => $file->getClientOriginalName(),
                        'file_path' => $path,
                        'mime_type' => $file->getMimeType(),
                        'size' => $file->getSize(),
                    ]);
                }
            }

            DB::commit();

            return redirect()
                ->route('plaintes.resolutions.show', [$plainte, $resolution])
                ->with('success', 'La proposition de résolution a été mise à jour avec succès.');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Erreur lors de la mise à jour de la résolution: ' . $e->getMessage());
            
            return redirect()
                ->back()
                ->with('error', 'Une erreur est survenue lors de la mise à jour de la proposition de résolution.')
                ->withInput();
        }
    }

    /**
     * Accept the specified resolution.
     */
    public function accept(Request $request, Plainte $plainte, Resolution $resolution)
    {
        $this->authorize('accept', $resolution);
        
        $request->validate([
            'commentaire' => 'nullable|string|max:1000',
        ]);

        DB::beginTransaction();
        try {
            $resolution->accept($request->commentaire);
            
            // Update plainte status if needed
            $plainte->update([
                'statut' => 'resolu',
                'date_resolution' => now(),
            ]);

            // Notify relevant parties
            $this->notifyResolutionStatusUpdate($resolution, 'acceptée');

            DB::commit();

            return redirect()
                ->route('plaintes.show', $plainte)
                ->with('success', 'La résolution a été acceptée avec succès.');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Erreur lors de l\'acceptation de la résolution: ' . $e->getMessage());
            
            return redirect()
                ->back()
                ->with('error', 'Une erreur est survenue lors de l\'acceptation de la résolution.');
        }
    }

    /**
     * Reject the specified resolution.
     */
    public function reject(Request $request, Plainte $plainte, Resolution $resolution)
    {
        $this->authorize('reject', $resolution);
        
        $request->validate([
            'commentaire' => 'required|string|min:10|max:1000',
        ]);

        DB::beginTransaction();
        try {
            $resolution->reject($request->commentaire);
            
            // Update plainte status if needed
            $plainte->update([
                'statut' => 'en_traitement',
            ]);

            // Notify relevant parties
            $this->notifyResolutionStatusUpdate($resolution, 'refusée');

            DB::commit();

            return redirect()
                ->route('plaintes.show', $plainte)
                ->with('success', 'La résolution a été refusée avec succès.');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Erreur lors du refus de la résolution: ' . $e->getMessage());
            
            return redirect()
                ->back()
                ->with('error', 'Une erreur est survenue lors du refus de la résolution.');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Plainte $plainte, Resolution $resolution)
    {
        $this->authorize('delete', $resolution);
        
        DB::beginTransaction();
        try {
            // Delete associated media files
            foreach ($resolution->media as $media) {
                Storage::disk('public')->delete($media->file_path);
                $media->delete();
            }

            $resolution->delete();

            DB::commit();

            return redirect()
                ->route('plaintes.show', $plainte)
                ->with('success', 'La proposition de résolution a été supprimée avec succès.');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Erreur lors de la suppression de la résolution: ' . $e->getMessage());
            
            return redirect()
                ->back()
                ->with('error', 'Une erreur est survenue lors de la suppression de la proposition de résolution.');
        }
    }

    /**
     * Notify relevant parties about resolution status update.
     */
    protected function notifyResolutionStatusUpdate(Resolution $resolution, string $status)
    {
        $plainte = $resolution->plainte;
        
        // Notify the person who proposed the resolution
        if ($resolution->proposer && $resolution->proposer->email) {
            Mail::to($resolution->proposer->email)
                ->send(new ResolutionStatusUpdatedMail($resolution, $status));
        }

        // Notify the complainant
        if ($plainte->plaignant && $plainte->plaignant->email) {
            Mail::to($plainte->plaignant->email)
                ->send(new ResolutionStatusUpdatedMail($resolution, $status));
        }
    }
}